<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class ContentSecurityPolicy
{
    public function handle(Request $request, Closure $next): Response
    {
        // 🔒 Redirect HTTP to HTTPS (if request not secure)
        if (!$request->isSecure() && app()->environment('local')) {
            return redirect()->secure($request->getRequestUri(), 301);
        }

        $response = $next($request);

        // Optional CSP Nonce
        $nonce = app()->make('view')->getShared()['csp_nonce'] ?? bin2hex(random_bytes(16));

        // CSP Policy (customize if needed)
        $csp = implode(' ', [
            "default-src 'self';",
            "script-src 'self' 'nonce-{$nonce}' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com https://www.googletagmanager.com https://www.google-analytics.com https://www.google.com;",
            "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com https://cdn.jsdelivr.net https://cdnjs.cloudflare.com https://www.google.com;",
            "font-src 'self' https://fonts.gstatic.com;",
            "img-src 'self' data: https://www.google-analytics.com;",
            "connect-src 'self' https://www.google-analytics.com;",
            "media-src 'self';",
            "object-src 'none';",
            "base-uri 'none';",
            "frame-ancestors 'none';",
            "form-action 'self';",
            
        ]);

        // Security Headers
        $response->headers->set('Content-Security-Policy', $csp);
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('X-Frame-Options', 'DENY');
        $response->headers->set('X-XSS-Protection', '1; mode=block');
        $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');

        // 🔒 HSTS (force HTTPS in browsers)
        $response->headers->set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');

        // Force only 'en-US' as language
        $response->headers->set('Content-Language', 'en-US');

        // Informative encoding header (note: does not actually compress response)
        $response->headers->set('Accept-Encoding', 'gzip');

        // Remove Server Signature
        $response->headers->remove('X-Powered-By');
        $response->headers->remove('Server');

        return $response;
    }
}
