<?php

namespace App\Http\Controllers\Admin\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SmasindiaLogin;
use App\Models\OtpLogin;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class OtpLoginController extends Controller
{


    // ... (rest of the class definition)

    public function showLoginForm()
    {
        if (session('admin_id')) {
            // Use the named route for the dashboard
            return redirect()->route('dashboard'); 
        }
        
        return view('Admin.auth.login');
    }

    // Handles the request to send the OTP (Forget Password or Initial Login)
    public function sendOtp(Request $request)
    {
        // 1. Validate Email and Password
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        $user = SmasindiaLogin::where('email', $request->email)->first();

        // 2. Check Credentials (Updated without Auth::guard)
        if (!$user) {
            return back()->withErrors(['email' => 'Invalid email or password.']);
        }

        // Manual password check using Hash
        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['email' => 'Invalid email or password.']);
        }
        // Credentials verified, now generate OTP

        // 3. Generate and Store OTP
        $otp = Str::upper(Str::random(3)) . random_int(100, 999);
        $otp = substr(str_shuffle($otp), 0, 6);

        OtpLogin::updateOrCreate(
            ['user_id' => $user->id],
            [
                'otp_login' => $otp,
                'expire_at' => Carbon::now()->addMinutes(2),
            ]
        );

        // 4. Send Email using Blade File (Modified Recipient)
        // try {
        //     Mail::send('Admin.emails.otp_template', ['otp' => $otp, 'user' => $user], function ($message) use ($user) {

        //         // Hardcoded recipient for testing, as requested
        //         $message->to('reach_us@smasindia.com')
        //             ->subject('OTP Test: Login for ' . $user->email);
        //     });
        // } catch (\Exception $e) {
        //     Log::error('OTP Email failed to send: ' . $e->getMessage());
        //     return back()->withErrors(['email' => 'Could not send OTP. Please try again.']);
        // }

        // 5. Redirect to OTP Verification
        $request->session()->put('otp_email', $user->email);

        return redirect()->route('otp.verify')->with('success', "OTP has been sent. It expires in 2 minutes.");
    }
    
    
    
    // public function sendOtp(Request $request)
    // {
    //     $request->validate([
    //         'email' => 'required|email',
    //         'password' => 'required'
    //     ]);
    
    //     // Check if admin exists
    //     $admin = SmasindiaLogin::where('email', $request->email)->first();
    
    //     if (!$admin) {
    //         $admin = SmasindiaLogin::create([
    //             'email' => $request->email,
    //             'username' => explode('@', $request->email)[0], 
    //             'password' => Hash::make($request->password)
    //         ]);
    //     } else {

    //         if (!Hash::check($request->password, $admin->password)) {
    //             return back()->with('error', 'Invalid email or password.');
    //         }
    //     }
    
    //  return redirect()->route('otp.verify')->with('success', "OTP has been sent. It expires in 2 minutes.");
    // }


    public function showOtpForm()
    {
        // 1. Check if the user is ALREADY LOGGED IN
        // If 'admin_id' is present, the user has completed OTP and is authenticated. Redirect to dashboard.
        if (session('admin_id')) {
            return redirect()->route('dashboard'); 
        }
        
        // 2. Check if the OTP flow has been initiated
        // If 'otp_email' is NOT present, the user skipped the login step. Redirect to login form.
        if (!session('otp_email')) {
            return redirect()->route('login');
        }
    
        // --- OTP PROCESSING BEGINS ---
        $user = SmasindiaLogin::where('email', session('otp_email'))->first();
    
        if (!$user) {
            // Clear session and redirect if user unexpectedly vanished
            session()->forget('otp_email');
            return redirect()->route('login');
        }
    
        // Fetch the OTP record for this user
        // $otpRecord = OtpLogin::where('user_id', $user->id)->first();
    
        // if ($otpRecord) {
        //     $expire_at = $otpRecord->expire_at;
        //     $remainingTime = Carbon::now()->diffInSeconds($expire_at, false);
        //     $remainingTime = max($remainingTime, 0);
        // } else {
        //     $remainingTime = 0;
        // }
        
        
        $otpRecord = OtpLogin::where('user_id', $user->id)->first();
        $remainingTime = $otpRecord ? Carbon::now()->diffInSeconds($otpRecord->expire_at, false) : 0;
    
        return view('Admin.auth.verify_otp', compact('remainingTime'));
    }


    public function logout(Request $request)
    {
        // Auth::logout(); // This is correctly removed as per your request
    
        // 👇 Ensure this line exists to clear the manual login state
        $request->session()->forget('admin_id'); 
        
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect('/');
    }




    // Handles OTP verification and Login
    public function verifyOtp(Request $request)
    {
        $request->validate(['otp' => 'required|size:6']);
        $email = session('otp_email');
    
        if (!$email) {
            return redirect()->route('login')->withErrors(['error' => 'Session expired. Please try again.']);
        }
    
        $user = SmasindiaLogin::where('email', $email)->first();
    
        if (!$user) {
            return redirect()->route('login');
        }
    
        $otpRecord = OtpLogin::where('user_id', $user->id)
            ->where('otp_login', $request->otp)
            ->where('expire_at', '>', Carbon::now())
            ->first();
    
        if (!$otpRecord) {
            return back()->withErrors(['otp' => 'Invalid or expired OTP.']);
        }
    
        // 🚩 CRITICAL CHANGE: Manual Login by setting the required session key
        // This replaces Auth::login($user) and satisfies the AdminAuth middleware.
        $request->session()->put('admin_id', $user->id); 
    
    
        // Regenerate session for security
        $request->session()->regenerate();
    
        return redirect()->intended(route('dashboard'));
    }

    // --- New Function: Show Forget Password Form ---
    public function showForgetPasswordForm()
    {
        // You can reuse the login view or create a specific one
        return view('Admin.auth.forgot_password');
    }

    // --- New Function: Send OTP for Password Reset (Uses existing logic) ---
    // Note: This function assumes your system does not require the user to enter their old password
    // when initiating a password reset, only their email.
    public function sendOtpForPasswordReset(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        $user = SmasindiaLogin::where('email', $request->email)->first();

        if (!$user) {
            // Send a generic success message even if the user isn't found to prevent enumeration attacks
            return redirect()->route('otp.verify')
                ->with('success', 'If an account exists, an OTP has been sent to your email.');
        }

        // --- Duplicate/Refactor OTP Generation and Mail Logic ---
        $otp = Str::upper(Str::random(3)) . random_int(100, 999);
        $otp = substr(str_shuffle($otp), 0, 6);

        OtpLogin::updateOrCreate(['user_id' => $user->id], ['otp_login' => $otp, 'expire_at' => Carbon::now()->addMinutes(2)]);

        // try {
        //     Mail::send('emails.otp_template', ['otp' => $otp, 'user' => $user], function ($message) use ($user) {
        //         // Hardcoded recipient for testing, as requested earlier
        //         $message->to('reach_us@smasindia.com')
        //             ->subject('Password Reset OTP');
        //     });
        // } catch (\Exception $e) {
        //     Log::error('Password Reset OTP Email failed: ' . $e->getMessage());
        //     return back()->withErrors(['email' => 'Could not send OTP. Please try again.']);
        // }

        $request->session()->put('otp_email', $user->email);

        return redirect()->route('otp.verify')
            ->with('success', "A password reset OTP has been sent to your email. It expires in 2 minutes.");
    }


    public function resendOtp(Request $request)
    {
        if (session('admin_id')) {
            return redirect()->route('dashboard'); 
        }
        
        $email = session('otp_email');
    
        if (!$email) {
            return redirect()->route('login')->withErrors(['error' => 'Your session expired. Please start over.']);
        }
    
        $user = SmasindiaLogin::where('email', $email)->first();
    
        if (!$user) {
            return redirect()->route('login');
        }
    
        // Fetch the OTP record for this user
        $otpRecord = OtpLogin::where('user_id', $user->id)->first();
    
        // If OTP hasn't expired, don't allow resend yet
        if ($otpRecord && Carbon::now()->lt($otpRecord->expire_at)) {
            return back()->withErrors(['error' => 'You can resend OTP after the current OTP expires.']);
        }
    
        // Generate new OTP and set expiration
        $otp = Str::upper(Str::random(3)) . random_int(100, 999);
        $otp = substr(str_shuffle($otp), 0, 6);
    
        OtpLogin::updateOrCreate(['user_id' => $user->id], ['otp_login' => $otp, 'expire_at' => Carbon::now()->addMinutes(2)]);
    
        // try {
        //     Mail::send('emails.otp_template', ['otp' => $otp, 'user' => $user], function ($message) use ($user) {
        //         $message->to('reach_us@smasindia.com')
        //             ->subject('Resend OTP Request');
        //     });
        // } catch (\Exception $e) {
        //     Log::error('Resend OTP Email failed: ' . $e->getMessage());
        //     return back()->withErrors(['error' => 'Could not resend OTP.']);
        // }
    
        return back()->with('success', 'A new OTP has been sent to your email. It expires in 2 minutes.');
    }
    
    
}
